/* -*-c++-*- 
 * This source code is proprietary of ADIT
 * Copyright (C) 2013 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Vadiraj Kaamsha <vadiraj.kaamsha@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

//#define DEBUG_FONT_TEXTURES
#include <limits.h>
#include <osg/Depth>
#include <algorithm>
#include <osg/FrameBufferObject>
#include <osgUtil/Statistics>
#include <osgBatchedText/BatcherBase>
#include <osgBatchedText/TextSimpleBase>

using namespace osgText;
using namespace osgBatchedText;

const char* BatchGeodeBase::_vertSourceStraightOrtho = {
   "attribute vec4 a_TransVecOutline;\n"
   "varying  vec4 VertexColor;\n"
   "varying  vec2 TexCoord0;\n"
   "varying float texValue;\n"

   "void main()\n"
   "{\n"
   "  mat4 mvmat = mat4(1.0);\n "
   "  mvmat[3][0]=a_TransVecOutline.x;\n"
   "  mvmat[3][1]=a_TransVecOutline.y;\n"
   "  mvmat[3][2]=a_TransVecOutline.z;\n"

   "  texValue = a_TransVecOutline.w;\n"
   "  gl_Position = gl_ProjectionMatrix * gl_ModelViewMatrix * mvmat * vec4( gl_Vertex.xyz, 1.0 );\n"
   "  VertexColor = gl_Color;\n"

   "  TexCoord0 = gl_MultiTexCoord0.xy;\n"
   "}\n"
};

const char* BatchGeodeBase::_vertSourceStraightScreen = {
   "varying vec4 VertexColor;\n"
   "varying vec2 TexCoord0;\n"
   "varying float texValue;\n"
   "uniform float u_meterToPointFactorNormalized;\n"
   "attribute vec4 a_TransVecOutline;\n"

   "void calcScaleMatrix( inout mat4 scaleMat )\n"
   "{\n"
   "  float distw = length(a_TransVecOutline.xyz);\n"
   "  float scalingFactor = abs(distw * u_meterToPointFactorNormalized);\n"
   "  scaleMat[0][0] = scalingFactor;\n"
   "  scaleMat[1][1] = scalingFactor;\n"
   "}\n"

   "void main()\n"
   "{\n"
   "  mat4 mvmat = mat4(1.0);\n"
   "  calcScaleMatrix( mvmat );\n"

   "  mvmat[3][0] = a_TransVecOutline.x;\n"
   "  mvmat[3][1] = a_TransVecOutline.y;\n"
   "  mvmat[3][2] = a_TransVecOutline.z;\n"

   "  texValue = a_TransVecOutline.w;\n"
   "  VertexColor = vec4(gl_Color.rgb, 1.0);\n"
   "  gl_Position = gl_ProjectionMatrix * mvmat * vec4( gl_Vertex.xyz, 1.0 );\n"

   "  TexCoord0 = gl_MultiTexCoord0.xy;\n"
   "}\n"
};


const char* BatchGeodeBase::_vertSourceCurvedGroundFixed = {
   "varying vec4 VertexColor;\n"
   "varying vec2 TexCoord0;\n"
   "varying float texValue;\n"
   "uniform float u_meterToPointFactorNormalized;\n"
   "attribute vec4 a_TransVecOutline;\n"
   "attribute vec3 a_RelOffset;\n"


   "void main()\n"
   "{\n"
   "  float distw = length(a_TransVecOutline.xyz);\n"

   "  mat4 mvmat = gl_ModelViewMatrix;\n"
   "  mvmat[3][0] = a_TransVecOutline.x;\n"
   "  mvmat[3][1] = a_TransVecOutline.y;\n"
   "  mvmat[3][2] = a_TransVecOutline.z;\n"

   "  texValue = a_TransVecOutline.w;\n"
   "  VertexColor = vec4(gl_Color.rgb, 1.0);\n"
   "  vec4 finalPos = vec4(a_RelOffset + (gl_Vertex.xyz * abs( distw * u_meterToPointFactorNormalized )), 1.0 );\n"
   "  gl_Position = gl_ProjectionMatrix * mvmat * finalPos;\n"

   "  TexCoord0 = gl_MultiTexCoord0.xy;\n"
   "}\n"
};


const char* BatchGeodeBase::_vertSourceCurvedScreen = {
   "varying vec4 VertexColor;\n"
   "varying vec2 TexCoord0;\n"
   "varying float texValue;\n"
   "uniform float u_meterToPointFactorNormalized;\n"
   "attribute vec4 a_TransVecOutline;\n"
   "attribute vec3 a_RelOffset;\n"

   "void calcScaleMatrix( inout mat4 scaleMat )\n"
   "{\n"
   "  float distw = length(a_TransVecOutline.xyz);\n"
   "  float scalingFactor = abs( distw * u_meterToPointFactorNormalized );\n"
   "  scaleMat[0][0] = scalingFactor;\n"
   "  scaleMat[1][1] = scalingFactor;\n"
   "}\n"

   "void main()\n"
   "{\n"
   "  mat4 mvmat = gl_ModelViewMatrix;\n"
   "  mvmat[3][0] = a_TransVecOutline.x;\n"
   "  mvmat[3][1] = a_TransVecOutline.y;\n"
   "  mvmat[3][2] = a_TransVecOutline.z;\n"

   "  mat4 TransScalingMat = mat4(1.0);\n"
   "  calcScaleMatrix( TransScalingMat );\n"
   "  TransScalingMat[3][0] = a_TransVecOutline.x;\n"
   "  TransScalingMat[3][1] = a_TransVecOutline.y;\n"
   "  TransScalingMat[3][2] = a_TransVecOutline.z;\n"

   "  texValue = a_TransVecOutline.w;\n"
   "  VertexColor = vec4(gl_Color.rgb, 1.0);\n"
   "  gl_Position = gl_ProjectionMatrix * (mvmat * vec4(a_RelOffset, 0.0) + TransScalingMat * vec4( gl_Vertex.xyz, 1.0 ));\n"

   "  TexCoord0 = gl_MultiTexCoord0.xy;\n"
   "}\n"
};

const char* BatchGeodeBase::_fragSource = {
#ifdef OSG_GLES2_AVAILABLE
   "precision mediump float;\n"
#endif
   "varying vec4 VertexColor;\n"
   "varying vec2 TexCoord0;\n"
   "varying float texValue;\n"
   "uniform sampler2D fontTextures[2]; // Index 0 represents the inner texture and index 1 represents the outer texture \n"
   "void main()\n"
   "{\n"
   "  vec4 IC = vec4(VertexColor.rgb, VertexColor.a * texture2D(fontTextures[0], TexCoord0).a);\n"
   "  vec4 OC = vec4(VertexColor.rgb, VertexColor.a * texture2D(fontTextures[1], TexCoord0).a);\n"
   "  gl_FragColor = mix(IC, OC, texValue);\n"
   "}\n"
};

GlyphTextureSimple::GlyphTextureSimple(BatcherBase* BatcherBase, FontTextureType textureType)
: _batcherBase(BatcherBase)
, _textureType(textureType)
, _margin(1)
{
   setWrap(WRAP_S, CLAMP_TO_EDGE);
   setWrap(WRAP_T, CLAMP_TO_EDGE);
   if (true == BatcherBase->isMipmapActive())
   {
      setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
   }
   else
   {
      setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
   }
   setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
   setMaxAnisotropy(8.0f);
}

GlyphTextureSimple::~GlyphTextureSimple()
{
   _batcherBase = NULL;
}

void GlyphTextureSimple::apply(osg::State& state) const
{
   if (NULL == _batcherBase->getTextureObject(_textureType))
   {
      {
         Texture2D::apply(state);
      }

      unsigned int context = state.getContextID();
      _batcherBase->setTextureObject(getTextureObject(context), _textureType);
   }
}

BatchGeodeBase::BatchGeodeBase() : _renderBin(0)
{
}

BatchGeodeBase::BatchGeodeBase(int renderBin) : _renderBin(renderBin)
{
   addUpdateCallback(new BatchUpdateCallBack);
}

void BatchGeodeBase::init(BatchDrawableBase::tShaderType shaderType, bool enableDepthTest)
{
   osg::StateSet* ss = getOrCreateStateSet();
   if (ss != NULL)
   {
      ss->setRenderBinDetails(_renderBin, "RenderBin");
      createShaderProgram(shaderType);

      ss->setMode(GL_DEPTH_TEST, (enableDepthTest ? osg::StateAttribute::ON : osg::StateAttribute::OFF) | osg::StateAttribute::OVERRIDE);

      if (_shaderProgram.valid())
      {
         ss->setAttribute(_shaderProgram);

         // Set up some per vertex attributes here.
         BatchDrawableBase::bindAttribLocations(_shaderProgram);
      }
   }
   setCullingActive(false);
}

BatchGeodeBase::BatchGeodeBase(const BatchGeodeBase& src, const osg::CopyOp& copyop)
  : Geode(src, copyop), _renderBin(src._renderBin), _shaderProgram(src._shaderProgram)
{
}


void BatchGeodeBase::BatchUpdateCallBack::operator()(osg::Node* node, osg::NodeVisitor* nv)
{
   if (nv->getVisitorType() == osg::NodeVisitor::UPDATE_VISITOR)
   {
      const osg::Geode::DrawableList& drawables = static_cast<BatchGeodeBase*>(node)->getDrawableList();

      for (osg::Geode::DrawableList::const_iterator itr = drawables.begin(); itr != drawables.end(); ++itr)
      {
         static_cast<BatchDrawableBase*>((*itr).get())->clearElementsToDraw();
      }
   }
   traverse(node, nv);
}

void BatchGeodeBase::createShaderProgram(BatchDrawableBase::tShaderType shaderType)
{
   _shaderProgram = new osg::Program;
   _shaderProgram->setName("BatchedTextShaderProg");

   _shaderProgram->addShader(new osg::Shader(osg::Shader::FRAGMENT, _fragSource));

   switch (shaderType)
   {
   case BatchDrawableBase::CURVED_GROUND_FIXED:
   case BatchDrawableBase::CURVED_GROUND:
      _shaderProgram->addShader(new osg::Shader(osg::Shader::VERTEX, _vertSourceCurvedGroundFixed));
      break;
   case BatchDrawableBase::CURVED_SCREEN:
      _shaderProgram->addShader(new osg::Shader(osg::Shader::VERTEX, _vertSourceCurvedScreen));
      break;
   case BatchDrawableBase::STRAIGHT_ORTHO:
      _shaderProgram->addShader(new osg::Shader(osg::Shader::VERTEX, _vertSourceStraightOrtho));
      break;
   case BatchDrawableBase::STRAIGHT_SCREEN:
   default:
      _shaderProgram->addShader(new osg::Shader(osg::Shader::VERTEX, _vertSourceStraightScreen));
   }
}

BatcherBase::BatcherBase(const FontDescr& fontDescr, bool useHalfFloat, bool outline, GLint maxTextureSize, unsigned int maxGlyphs, bool mipmapActive) :
    GlyphInfoContainerBase(fontDescr, outline, maxGlyphs),
#ifdef USE_COMBINED_SUBLOAD
    _data(NULL),
    _combined_subload(false),
#endif
    _texture_modified(false),
    _max_retries(20),
    _height_factor(1),
    _initailizeTexture(true),
    _view(NULL),
    _sortBasedOnDisplayOrder(false),
    _useHalfFloat(useHalfFloat),
    _mipmapActive(mipmapActive)
{
   _textureSize._texture_width = maxTextureSize;
   _textureSize._texture_height = _textureSize._texture_width;

   init();
}

BatcherBase::BatcherBase(const BatcherBase& src):
   GlyphInfoContainerBase(src),
   osg::Referenced(src),
#ifdef USE_COMBINED_SUBLOAD
    _data(NULL),
    _combined_subload(false),
#endif
    _texture_modified(false),
    _max_retries(src._max_retries),
    _height_factor(src._height_factor),
    _initailizeTexture(true),
    _view(NULL),
    _sortBasedOnDisplayOrder(src._sortBasedOnDisplayOrder),
    _useHalfFloat(src._useHalfFloat),
    _mipmapActive(src._mipmapActive)
{
   _textureSize._texture_width = src._textureSize._texture_width;
   _textureSize._texture_height = _textureSize._texture_width;

   init();
}

BatcherBase::~BatcherBase()
{
   _textureObject[E_INNER_FONT_TEXTURE] = NULL;
   _textureObject[E_OUTER_FONT_TEXTURE] = NULL;
   _view = NULL;

   for (std::list<BatchEntry*>::iterator itr = _deleteFromTexture.begin(); itr != _deleteFromTexture.end(); ++itr)
   {
      delete *itr;
   }

#ifdef USE_COMBINED_SUBLOAD
   free(_data);
   _data = NULL;
#endif   //USE_COMBINED_SUBLOAD
}

void BatcherBase::init()
{
   _textureObject[E_INNER_FONT_TEXTURE] = NULL;
   _textureObject[E_OUTER_FONT_TEXTURE] = NULL;

#ifdef USE_COMBINED_SUBLOAD
   _data = (unsigned char*)calloc(_textureSize._texture_width * _textureSize._texture_height, sizeof(unsigned char));
#endif   //USE_COMBINED_SUBLOAD

   _newFontStateSet = new osg::StateSet;
   if (_newFontStateSet.valid())
   {
      _fontTexture[E_INNER_FONT_TEXTURE] = new GlyphTextureSimple(this, E_INNER_FONT_TEXTURE);

      if (_fontTexture[E_INNER_FONT_TEXTURE].valid() || _outline)
      {
         osg::ref_ptr<osg::Uniform> textureUniform = _newFontStateSet->getOrCreateUniform("fontTextures", osg::Uniform::SAMPLER_2D, E_COUNT_FONT_TEXTURE);
         textureUniform->setElement(0, E_INNER_FONT_TEXTURE);
         textureUniform->setElement(1, E_OUTER_FONT_TEXTURE);
      }

      if (_fontTexture[E_INNER_FONT_TEXTURE].valid())
      {
         _fontTexture[E_INNER_FONT_TEXTURE]->setTextureSize(_textureSize._texture_width, _textureSize._texture_height);
         _fontTexture[E_INNER_FONT_TEXTURE]->setInternalFormat(GL_ALPHA);

         _newFontStateSet->setTextureAttribute(E_INNER_FONT_TEXTURE, _fontTexture[E_INNER_FONT_TEXTURE], osg::StateAttribute::OFF);
      }

      if (_outline)
      {
         _height_factor = 2;
         //Create texture for outline
         _fontTexture[E_OUTER_FONT_TEXTURE] = new GlyphTextureSimple(this, E_OUTER_FONT_TEXTURE);
         if (_fontTexture[E_OUTER_FONT_TEXTURE].valid())
         {
            _fontTexture[E_OUTER_FONT_TEXTURE]->setTextureSize(_textureSize._texture_width, _textureSize._texture_height);
            _fontTexture[E_OUTER_FONT_TEXTURE]->setInternalFormat(GL_ALPHA);

            _newFontStateSet->setTextureAttribute(E_OUTER_FONT_TEXTURE, _fontTexture[E_OUTER_FONT_TEXTURE], osg::StateAttribute::OFF);
         }
      }

      //the write mask is set to false in BatchDrawableBase::draw, because otherwise rendering of object with several textures does not work correctly
      // e.g. font rendering with outline: here filling and outline have same z-value;
      // font with a background image: here also both parts have same z-value
      // so the correct drawing order can't be achieved if depth write is on, but only keeping the drawing order
      //osg::ref_ptr<osg::Depth> depth = new osg::Depth;
      //if (depth.valid())
      //{
      //   depth->setWriteMask(false);
      //   _newFontStateSet->setAttributeAndModes(depth, osg::StateAttribute::ON);
      //}
   }
}

void BatcherBase::deleteTextures() const
{
   if (_textureObject[E_INNER_FONT_TEXTURE] != NULL)
   {
      glDeleteTextures(1, &_textureObject[E_INNER_FONT_TEXTURE]->_id);
   }
   if (_textureObject[E_OUTER_FONT_TEXTURE] != NULL)
   {
      glDeleteTextures(1, &_textureObject[E_OUTER_FONT_TEXTURE]->_id);
   }
}


BatchDrawableBase* BatcherBase::getOrCreateBatchDrawable(int renderBin, BatchDrawableBase::tShaderType shaderType, bool enableDepthTest)
{
   std::map< int, osg::ref_ptr<BatchDrawableBase> >::iterator itr = _renderBinVsBatchDrawable.find(renderBin);
   if (itr == _renderBinVsBatchDrawable.end())
   {
      osg::ref_ptr<BatchDrawableBase> drawableInfo = new BatchDrawableBase(this, shaderType, enableDepthTest);
      itr = _renderBinVsBatchDrawable.insert(std::make_pair(renderBin, drawableInfo)).first;
   }
   return itr->second.get();
}


void BatcherBase::addBatchElementContainer(const osg::ref_ptr<BatchElementContainer>& batchElementContainer, int renderBin, const osg::View* view, bool update)
{
   if (_view != view)
   {
      clearBatchElementVectorsToDraw();
      _view = view;
   }

   getOrCreateBatchDrawable(renderBin, batchElementContainer->_renderInfo._shaderType, batchElementContainer->_renderInfo._enableDepthTest)->addBatchElementContainer(batchElementContainer, update);
}

int BatcherBase::getDefaultRenderBin(BatchDrawableBase::tShaderType shaderType)
{
   int renderBin = 1;
   switch (shaderType)
   {
   case BatchDrawableBase::CURVED_GROUND_FIXED:
      renderBin = 2;
      break;
   case BatchDrawableBase::STRAIGHT_GROUND_FIXED:
      renderBin = 3;
      break;
   case BatchDrawableBase::CURVED_GROUND:
      renderBin = 4;
      break;
   case BatchDrawableBase::STRAIGHT_GROUND:
      renderBin = 5;
      break;
   case BatchDrawableBase::CURVED_SCREEN:
      renderBin = 6;
      break;
   case BatchDrawableBase::STRAIGHT_SCREEN:
      renderBin = 7;
      break;
   case BatchDrawableBase::STRAIGHT_ORTHO:
   default:
      renderBin = 8;
   }
   return renderBin;
}

void BatcherBase::addForDeletion(BatchEntry* batchEntry) const
{
   if (batchEntry)
   {
      if (batchEntry->_glyph._lower._inTexture)
      {
         _deleteFromTexture.push_back(batchEntry);
      }
      else
      {
         delete batchEntry;
      }
   }
}

BatchEntry* BatcherBase::findOrAddGlyph(bool isCharCode, const unsigned int charCodeOrGlyphIndex, bool incrementCounter) const
{
   if (incrementCounter)
   {
      ++_counter;
   }

   return addGlyphToCache(isCharCode, charCodeOrGlyphIndex);
}

bool BatcherBase::allocateSpaceForGlyph(osgText::glyphEntry &glyph, osg::State& state) const
{
    if (glyph._lower._inTexture)
    {
        return true;
    }

    std::vector<textureRow>::iterator row;
    Texturecols::iterator col;
    unsigned int width  = static_cast<unsigned int>(glyph._upper._dimension.x());
    unsigned int height = static_cast<unsigned int>(-glyph._upper._dimension.y());
    unsigned int posX = 0;
    unsigned int posY = 0;
    bool added = false;

    if ((0 == width) || (0 == height))
    {
        static const osg::Vec2 dummy;
        glyph._lower._posX = 0;
        glyph._lower._posY = 0;
        /* indexing is done in state.drawelments */
        glyph._lower._texCoord.clear();
        glyph._lower._texCoord.reserve(4);
        glyph._lower._texCoord.push_back(dummy);
        glyph._lower._texCoord.push_back(dummy);
        glyph._lower._texCoord.push_back(dummy);
        glyph._lower._texCoord.push_back(dummy);
        glyph._lower._inTexture = true;
        return true;
    }

    if(_rows.empty())
    {
        if (NULL != _textureObject[E_INNER_FONT_TEXTURE])
        {
           unsigned char* data = (unsigned char*)calloc(_textureSize._texture_width * _textureSize._texture_height, sizeof(unsigned char));
           state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
           glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
           glTexImage2D( GL_TEXTURE_2D, 0, GL_ALPHA,
              _textureSize._texture_width, _textureSize._texture_height, 0,
              GL_ALPHA,
              GL_UNSIGNED_BYTE,
              data );

           if (_outline && (NULL != _textureObject[E_OUTER_FONT_TEXTURE]))
           {
              state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
              glBindTexture(GL_TEXTURE_2D, _textureObject[E_OUTER_FONT_TEXTURE]->_id);
              glTexImage2D( GL_TEXTURE_2D, 0, GL_ALPHA,
                 _textureSize._texture_width, _textureSize._texture_height, 0,
                 GL_ALPHA,
                 GL_UNSIGNED_BYTE,
                 data );
           }

           free(data);
           data = NULL;
        }

        _textureSize._inv_texture_width  = 1.0f / static_cast<float>(_textureSize._texture_width);
        _textureSize._inv_texture_height = 1.0f / static_cast<float>(_textureSize._texture_height);

        textureCol aCol(0, _textureSize._texture_width - (width + 2));
        Texturecols cols;
        cols.push_back(aCol);
        textureRow aRow(0, (height + 2), false, cols);
        _rows.push_back(aRow);
        posX = aCol._free_width + 1;
        posY = 1;
        added = true;
    }
    else
    {
        for(row = _rows.begin(); ((row != _rows.end()) && !added); ++row)
        {
            if((!row->_fixed && ((row->_y + (height + 2)) < _textureSize._texture_height)) || ((height + 2) <= row->_height))
            {
                /* check for big enough space within column */
                for(col = row->_cols.begin(); ((col != row->_cols.end()) && !added); ++col)
                {
                    if(col->_free_width >= (width + 2))
                    {
                        posY = row->_y + 1;
                        posX = col->_x + (col->_free_width - (width + 2)) + 1;
                        col->_free_width -= (width + 2);
                        if(!row->_fixed && (row->_height < (height + 2)))
                        {
                            row->_height = (height + 2);
                        }
                        added = true;
                    }
                }
            }
        }
        /* glyph did not fit into existing rows/columns, so add a new empty row(and fix the height of the previous one) */
        if (!added)
        {
            /* add new row only when the remaining height of the texture is sufficient */
            textureRow& aRow_tmp = _rows.back();
            unsigned int new_y = aRow_tmp._y + aRow_tmp._height;
            if((new_y + height + 2) < _textureSize._texture_height)
            {
                textureCol aCol(0, _textureSize._texture_width - (width + 2));
                Texturecols cols;
                cols.push_back(aCol);
                textureRow aRow(new_y , (height + 2), false, cols);
                aRow_tmp._fixed = true;
                _rows.push_back(aRow);
                posX = aCol._free_width + 1;
                posY = aRow._y + 1;
                added = true;
            }
        }
    }
    if(added)
    {
        glyph._lower._inTexture = true;
        _texture_modified = true;
#ifdef USE_COMBINED_SUBLOAD
        if(_combined_subload)
        {
            /* copy glyph data to "texture representation */
            /* and delete glyph data to reserve ram */
            for(unsigned int loop = 0; loop < height; loop++)
            {
                memcpy(&glyph._lower._data[loop*width],&_data[((posY + loop) * _textureSize._texture_width) + posX], width);
            }
            /* in case of outline copy the "stroked" glyph to the upper half of the texture */
            if(_outline)
            {
                for(unsigned int loop = 0; loop < height; loop++)
                {
                    memcpy(&glyph._lower._data[(loop + height)*width],&_data[((posY + loop + _textureSize._texture_height) * _textureSize._texture_width) + posX], width);
                }
            }
            delete [] glyph._lower._data;
            glyph._lower._data = NULL;
        }
        else
#endif
        {
           if (NULL != _textureObject[E_INNER_FONT_TEXTURE])
           {
              glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
              state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
              glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
              glTexSubImage2D(GL_TEXTURE_2D, 0, posX, posY,
                              width, height,
                              GL_ALPHA, GL_UNSIGNED_BYTE, glyph._lower._data);

              if (_outline && (NULL != _textureObject[E_OUTER_FONT_TEXTURE]))
              {
                 state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
                 glBindTexture(GL_TEXTURE_2D, _textureObject[E_OUTER_FONT_TEXTURE]->_id);
                 glTexSubImage2D(GL_TEXTURE_2D, 0, posX, posY,
                                 width, height,
                                 GL_ALPHA, GL_UNSIGNED_BYTE, &glyph._lower._data[width * height]);
              }
           }
        }
        float tex_left   = static_cast<float>(posX) * _textureSize._inv_texture_width;
        float tex_bottom = static_cast<float>(posY) * _textureSize._inv_texture_height;
        float tex_right  = tex_left   + (static_cast<float>(width)  * _textureSize._inv_texture_width);
        float tex_up     = tex_bottom + (static_cast<float>(height) * _textureSize._inv_texture_height);

        osg::Vec2 lb(tex_left, tex_bottom);
        osg::Vec2 lu(tex_left, tex_up);
        osg::Vec2 ru(tex_right, tex_up);
        osg::Vec2 rb(tex_right, tex_bottom);

        glyph._lower._posX = posX;
        glyph._lower._posY = posY;
        /* indexing is done in state.drawelments */
        glyph._lower._texCoord.clear();
        glyph._lower._texCoord.reserve(4);
        glyph._lower._texCoord.push_back(lu);
        glyph._lower._texCoord.push_back(ru);
        glyph._lower._texCoord.push_back(rb);
        glyph._lower._texCoord.push_back(lb);
    }

    return added;
}

bool BatcherBase::releaseSpaceForGlyph(osgText::glyphEntry &glyph, osg::State& state) const
{
   std::vector<textureRow>::iterator row;
   Texturecols::iterator col;
   Texturecols::iterator next_col;
   bool deleted = false;
   unsigned int width  = static_cast<unsigned int>(glyph._upper._dimension.x()) + 2u;
#ifdef USE_COMBINED_SUBLOAD
   unsigned int height = static_cast<unsigned int>(-glyph._upper._dimension.y());
#endif
   unsigned int posX   = glyph._lower._posX - 1;
   unsigned int posY   = glyph._lower._posY - 1;
   textureCol aCol(posX, width);

   for(row = _rows.begin(); row != _rows.end(); ++row)
   {
      if(row->_y == posY)
      {
         break;
      }
   }

   if ( (row != _rows.end())
        && (NULL != _textureObject[E_INNER_FONT_TEXTURE]))
   {
      //Clear texture region
      unsigned int height = row->_height;

      unsigned char* data = (unsigned char*)calloc(width * height, sizeof(unsigned char));

      glPixelStorei(GL_UNPACK_ALIGNMENT,1);
      state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
      glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
      glTexSubImage2D( GL_TEXTURE_2D, 0, posX, posY,
         width, height,
         GL_ALPHA, GL_UNSIGNED_BYTE, data );

#ifdef DEBUG_FONT_TEXTURES
      static bool flag = false;
      if (flag)
         dumpTextures(_textureObject[E_INNER_FONT_TEXTURE]->_id, contextID);
#endif   //DEBUG_FONT_TEXTURES

      if (_outline && (NULL != _textureObject[E_OUTER_FONT_TEXTURE]))
      {
         state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
         glBindTexture(GL_TEXTURE_2D, _textureObject[E_OUTER_FONT_TEXTURE]->_id);
         glTexSubImage2D( GL_TEXTURE_2D, 0, posX, posY,
            width, height,
            GL_ALPHA, GL_UNSIGNED_BYTE, data );

#ifdef DEBUG_FONT_TEXTURES
         if (flag)
            dumpTextures(_textureObject[E_OUTER_FONT_TEXTURE]->_id, contextID);
#endif   //DEBUG_FONT_TEXTURES
      }
      free(data);
      data = NULL;

      //ME_TR_TYPES_INFO((__LINE__, __FILE__, "---------------------------[[[START]]]-------------------------"));
      for(col = row->_cols.begin(); ((col != row->_cols.end()) && (!deleted)); ++col)
      {
         /* previous free area can be combined with new free area */
         if(col->_x + col->_free_width == posX)
         {
            //ME_TR_TYPES_INFO((__LINE__, __FILE__, "First IF, x:%d, FW:%d, pX:%d, W:%d:", col->_x, col->_free_width, posX, width));
            col->_free_width += width;
            /* now check if next element can be combined with current */
            next_col = col;
            ++next_col;
            if(next_col != row->_cols.end())
            {
               if((col->_x + col->_free_width) == next_col->_x)
               {
                  col->_free_width += next_col->_free_width;
                  row->_cols.erase(next_col);
                  //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Erased"));
               }
            }
            deleted = true;
         }
         else
         {
            //ME_TR_TYPES_INFO((__LINE__, __FILE__, "ELSE"));

            next_col = col;
            ++next_col;

            /* next entry is bigger -> insert before */
            if(col->_x > posX)
            {
               row->_cols.insert(col, aCol);
               deleted = true;
               //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Inserted Before x:%d, FW:%d, pX:%d, W:%d", col->_x, col->_free_width, posX, width));
            }
            else if(next_col != row->_cols.end()) /* when there are next elements... */
            {
               //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Not Last"));
               /* next element can be combined with new free area */
               if( posX + width == next_col->_x)
               {
                  //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Merged x:%d, FW:%d, pX:%d, W:%d", next_col->_x, next_col->_free_width, posX, width));
                  next_col->_x = posX;
                  next_col->_free_width += width;
                  deleted = true;
               }
            }
            else
            {
               //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Last"));
               //textureCol& Col_last = row->_cols.back();
               //ME_TR_TYPES_INFO((__LINE__, __FILE__, "Added Last x:%d, FW:%d, pX:%d, W:%d", Col_last._x, Col_last._free_width, posX, width));

               /* all entries checked -> add a new entry to the end */
               row->_cols.insert(row->_cols.end(), aCol);
               deleted = true;
            }
         }
      }
      //ME_TR_TYPES_INFO((__LINE__, __FILE__, "[[[END]]]\n"));
   }

   if(deleted)
   {
      glyph._lower._inTexture = false;
      glyph._lower._texCoord.clear();
#ifdef USE_COMBINED_SUBLOAD
      if(_combined_subload)
      {
         glyph._lower._data = new unsigned char[width * height * _height_factor];
         /* copy glyph back from "texture representation to glyph data*/
         for(unsigned int loop = 0; loop < height; loop++)
         {
            memcpy(&glyph._lower._data[loop*width], &_data[((posY + loop) * _textureSize._texture_width) + posX], width);
            /* clean up texture representation */
            memset(&_data[((posY + loop) * _textureSize._texture_width) + posX], 0, width);
         }
         /* in case of outline copy the "stroked" glyph to the upper half of the texture */
         if(_outline)
         {
            for(unsigned int loop = 0; loop < height; loop++)
            {
               memcpy(&glyph._lower._data[(loop + height)*width], &_data[((posY + loop + _textureSize._texture_height) * _textureSize._texture_width) + posX], width);
               /* clean up texture representation */
               memset(&_data[((posY + loop + _textureSize._texture_height) * _textureSize._texture_width) + posX], 0, width);
            }
         }
      }
#endif

      textureRow& aRow_last = _rows.back();
      col = aRow_last._cols.begin();
      if((col->_free_width == _textureSize._texture_width) && (_rows.size() > 1))
      {
         _rows.pop_back();
         textureRow& aRow_temp = _rows.back();
         aRow_temp._fixed = false;
      }
   }

   return deleted;
}

void BatcherBase::updateUniforms(osg::State & state) const
{
   float nearPlaneDis = 0.0f, farPlaneDis = 0.0f, aspectR = 0.0f, fovy = 0.0f;

   if (state.getProjectionMatrix().getPerspective(fovy, aspectR, nearPlaneDis, farPlaneDis))
   {
      const osg::Viewport* vp = state.getCurrentViewport();

      //u_meterToPointFactorNormalized is the meter to point relation at 1m camera distance
      osg::Uniform* uniTF = _newFontStateSet->getOrCreateUniform("u_meterToPointFactorNormalized", osg::Uniform::FLOAT);
      if (uniTF && vp)
      {
         float transformationFactor = 2.0f * tanf(fovy * (static_cast<float>(osg::PI) * 0.5f / 180.0f)) / vp->height();
         uniTF->set(transformationFactor);
      }
   }
}

void BatcherBase::generateMipMap(osg::State& state) const
{
   if ( (_texture_modified)
        && (NULL != _textureObject[E_INNER_FONT_TEXTURE]))
   {
#ifdef USE_COMBINED_SUBLOAD
      if(_combined_subload)
      {
         state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
         glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
         glTexSubImage2D( GL_TEXTURE_2D, 0, 0, 0,
            _textureSize._texture_width, (_textureSize._texture_height * _height_factor),
            GL_ALPHA, GL_UNSIGNED_BYTE, _data );
      }
#endif

#if defined(OSG_GLES2_AVAILABLE)
      state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
      glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
      glGenerateMipmap(GL_TEXTURE_2D);
      if(_outline && (NULL != _textureObject[E_OUTER_FONT_TEXTURE]))
      {
         state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
         glBindTexture(GL_TEXTURE_2D, _textureObject[E_OUTER_FONT_TEXTURE]->_id);
         glGenerateMipmap(GL_TEXTURE_2D);
      }
#else
      osg::FBOExtensions* fbo_ext = osg::FBOExtensions::instance(state.getContextID(), true);
      if (fbo_ext)
      {
         state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
         glBindTexture(GL_TEXTURE_2D, _textureObject[E_INNER_FONT_TEXTURE]->_id);
         fbo_ext->glGenerateMipmap(_textureObject[E_INNER_FONT_TEXTURE]->target());

         if (_outline && (NULL != _textureObject[E_OUTER_FONT_TEXTURE]))
         {
            state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
            glBindTexture(GL_TEXTURE_2D, _textureObject[E_OUTER_FONT_TEXTURE]->_id);
            fbo_ext->glGenerateMipmap(_textureObject[E_OUTER_FONT_TEXTURE]->target());
         }
      }
#endif
   }
}


unsigned int BatcherBase::getNumberOfBatchedElements()
{
   unsigned int numberOfBatchedElements = 0;
   for (std::map<int, osg::ref_ptr<BatchDrawableBase> >::iterator itr = _renderBinVsBatchDrawable.begin(); itr != _renderBinVsBatchDrawable.end(); ++itr)
   {
      numberOfBatchedElements += itr->second->getNumberOfBatchedElements();
   }
   return numberOfBatchedElements;
}

void BatcherBase::clearBatchElementVectorsToDraw()
{
   for (std::map<int, osg::ref_ptr<BatchDrawableBase> >::iterator itr = _renderBinVsBatchDrawable.begin(); itr != _renderBinVsBatchDrawable.end(); ++itr)
   {
      itr->second->clearElementsToDraw();
   }
}

osg::ref_ptr<BatcherBase> BatcherBase::getOrCreateBatcherBase(const osg::ref_ptr<TextConfigBase>& textConfigBase, tBatcherBaseMap& BatcherBaseMap)
{
   osg::ref_ptr<BatcherBase> batchedText;

   tBatcherBaseMap::iterator actBT = BatcherBaseMap.find(textConfigBase->getFontDescr());

   if (actBT != BatcherBaseMap.end())
   {
      batchedText = actBT->second;
   }
   else
   {
      //create new textBatch
      batchedText = new BatcherBase(textConfigBase->getFontDescr());
      if (batchedText.valid())
      {
         batchedText->constructDefaultGlyphs();
         BatcherBaseMap.insert(tBatcherBasePair(textConfigBase->getFontDescr(), batchedText));
      }
   }

   return batchedText;
}

void BatcherBase::initializeTexture(osg::State& state) const
{
   if (_initailizeTexture)
   {
      GlyphMap::iterator itr;
      bool success = true;
      for (itr = _glyphMapOfGlyphIndices.begin(); (itr != _glyphMapOfGlyphIndices.end() && success); ++itr)
      {
         success = allocateSpaceForGlyph(itr->second->_glyph, state);
      }
      _initailizeTexture = false;
   }
}


void BatcherBase::releaseSpace(osg::State& state)
{
   bool result = false;
   for (std::list<BatchEntry*>::iterator releaseGlyphItr = _glyphContainer.begin(); (releaseGlyphItr != _glyphContainer.end()) && !result; ++releaseGlyphItr)
   {
      BatchEntry* releaseEntry = *releaseGlyphItr;
      if (releaseEntry->_glyph._lower._inTexture && releaseEntry->_charcode != 32 && releaseEntry->_counter < _counter)
      {
         result = releaseSpaceForGlyph(releaseEntry->_glyph, state);
      }
   }
}

void BatcherBase::releaseDeletedTextures(osg::State &state) const
{
   if (0 != _deleteFromTexture.size())
   {
      for (std::list<BatchEntry*>::iterator itr = _deleteFromTexture.begin(); itr != _deleteFromTexture.end(); ++itr)
      {
         //remove from texture
         if ((*itr)->_glyph._lower._inTexture)
         {
            releaseSpaceForGlyph((*itr)->_glyph, state);
         }
         delete *itr;
      }
      _deleteFromTexture.clear();
   }
}

osg::Drawable* BatcherBase::getDrawable(int renderBin)
{
   osg::Drawable* batchDrawable = NULL;
   std::map< int, osg::ref_ptr<BatchDrawableBase> >::const_iterator itr = _renderBinVsBatchDrawable.find(renderBin);
   if (itr != _renderBinVsBatchDrawable.end())
   {
      batchDrawable = itr->second.get();
   }
   return batchDrawable;
}

#ifdef DEBUG_FONT_TEXTURES
void BatcherBase::dumpTextures(GLuint tex_id, unsigned int contextID) const
{
   static unsigned int fbo_id = 0;
   if (fbo_id == 0)
   {
      osg::FBOExtensions* fbo_ext = osg::FBOExtensions::instance(contextID, true);
      if (fbo_ext)
      {
         fbo_ext->glGenFramebuffers(1, &fbo_id);
      }
   }
   if (fbo_id != 0)
   {
      osg::FBOExtensions* fbo_ext = osg::FBOExtensions::instance(contextID, true);
      if (fbo_ext)
      {
         fbo_ext->glBindFramebuffer(GL_FRAMEBUFFER_EXT, fbo_id);
         fbo_ext->glFramebufferTexture(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, tex_id, 0);
         glReadBuffer(GL_COLOR_ATTACHMENT0_EXT);

         static unsigned char* data = NULL;
         if (NULL == data)
         {
            data = new unsigned char[_textureSize._texture_width*_textureSize._texture_height];
         }
         glReadPixels(0, 0, _textureSize._texture_width, _textureSize._texture_height,
            GL_ALPHA, GL_UNSIGNED_BYTE, data);

         static int cntr = 0;
         cntr++;
         static char name[32] = {0};
         sprintf(name, "%d_%d.raw", tex_id, cntr);
         FILE* fp = fopen(name, "wb");
         if (NULL != fp)
         {
            fwrite(data, _textureSize._texture_width*_textureSize._texture_height, 1, fp);
            fclose(fp);
         }
      }
   }
}
#endif   //DEBUG_FONT_TEXTURES
